// uvi.c: Signetics 2637 UVI emulation module (platform-independent)

#ifdef AMIGA
    #include "lint.h"
    #include "amiga.h"
#endif
#ifdef WIN32
    #include "ibm.h"
#endif

#include "aa.h"
#include "uvi.h"

#include <stdio.h>
#include <string.h>

#define VLOCK_OFFSET (A_SPAREROWS / 2)  // centre

#define CPL           52 // CYCLESPERLINE is a reserved macro under Windows
// shouldn't this be about 57?
#define INPUTLINE    207
#define LATCHLINE    225
#define SENSELINE    240

#define FLIPS          2

IMPORT ULONG  analog,
              autofire,
              collisions,
              demultiplex,
              downframes,
              flagline,
              paused,
              swapped,
              totalframes,
              vlock,
              warp;
IMPORT ULONG  elapsed,
              frames;
IMPORT UBYTE* IOBuffer;
IMPORT FILE*  MacroHandle;
IMPORT TEXT   friendly[2 + 13 + 1 + 1];
IMPORT int    frameskip,
              offset,
              overcalc, // must be signed!
              rast,
              rastn,
              recmode,
              recsize,
              slice,
              wp;
IMPORT UBYTE  memory[32768],
              screen[BOXWIDTH][BOXHEIGHT],
              psu;
IMPORT UWORD  page,
              iar;
IMPORT FLAG   consoleopen,
              crippled,
              inframe,
              loop,
              runtorastline,
              runtodma,
              runtoframe,
              limitrefresh,
              trace;
IMPORT SLONG  ax[2], // analog paddle X-coords
              ay[2]; // analog paddle Y-coords

#ifdef WIN32
    #ifdef MODA_SHALLOW
        IMPORT UBYTE     display[EVENBOXWIDTH * BOXHEIGHT];
    #else
        IMPORT int       display[BOXWIDTH * BOXHEIGHT];
        IMPORT ULONG     pens[16];
    #endif
    IMPORT UBYTE         sx[2], sy[2];
#endif

#ifdef AMIGA
    IMPORT FLAG          fullscreen;
    IMPORT LONG          pens[9];
    IMPORT int           px, py,
                         wide,
                         size;
    IMPORT UBYTE*        lineptr[MAXBOXHEIGHT];
#endif

MODULE FLAG   theflag[BOXHEIGHT];  // Flag pin for this line (TRUE or FALSE)
MODULE UBYTE  bgcolour[BOXHEIGHT]; // background colour for this line (taking account of Flag pin)
MODULE SLONG  x,
              y,
              xx,
              yy;
MODULE FLAG   newband,
              skip[FLIPS][4];
MODULE UBYTE  // input
              result,
              t,
              bx,
              by,
              // graphics
              fgc,
              fgc1, fgc2,
              bgc = GREY,
              bgc1, bgc2,
              hires, // 0 for low resolution, 1 for high resolution
              spritefgc[FLIPS][4][16],
              image[FLIPS][4][8],
              sprcolour,
              boardmode,
              // macros
              OutputBuffer[11],
              rowbuf[16];
MODULE SLONG  hoffset, // = XMARGIN + 0..7
              soffset[FLIPS],
              spritex[FLIPS][4],
              spritey[FLIPS][4],
              rowc, // goes up only 1 per 2 rasts in low-res mode
                    // 0..7: which y-pixel row in the current "band" (row of chars) we are doing
              row,  // whenever rowc gets to 8, rowc is cleared and row is incremented
              blockmode,
              column,
              thechar,
              latch_a,
              latch_b,
              latch_start,
              tall[FLIPS][4], // formerly bxy
              whichsprite, // formerly bzs
              imagedata,
              rowlimit,
              flip = 0;
MODULE int    machinevoffset, // the voffset that the machine sees
              stopline;

/* Bottom margins are as follows (for initial screen):
 HOBO:     (very thick) yellow
 THE-END:  thick green, medium yellow, thin cyan
 TURTLES:  (very thick) green
 JUMPBUG?: thin green, thick yellow
These were tested under MESS V2, WinArcadia 1.5
*/

// MODULE FUNCTIONS

MODULE void a_playerinput(UBYTE source, UBYTE dest);
MODULE void drawsprites(SLONG flipper);
MODULE __inline void a_emuinput(void);
MODULE int sprite_collision(int first, int second);
MODULE void drawallsprites(void);
MODULE __inline void collidesprites(void);
MODULE __inline void specialdma(void);
MODULE __inline void breakdma(void);
MODULE __inline void breakrastline(void);
MODULE void uviwrite(int address, UBYTE data);

#ifdef USEFUNCPTRS
    IMPORT void (* drawpixel)(SLONG x, SLONG y, UBYTE colour);
#else
    MODULE __inline void drawpixel(SLONG x, SLONG y, UBYTE colour);
#endif

/* start of code

Test version of uvi() which measures graphics rendering speed.
EXPORT __inline void uvi(void)
{   for (bgc = 0; bgc <= 7; bgc++)
    {   for (y = 0; y < BOXHEIGHT; y++)
        {   for (x = 0; x < BOXWIDTH; x++)
            {   if (screen[x][y] != bgc)
                {   drawpixel(x, y, bgc);
        }   }   }
        updatescreen();
        frames++;
}   } */

EXPORT __inline void uvi(void)
{   PERSIST int uservoffset; // the voffset that the user sees

    // assert(machine == ARCADIA);

    inframe = TRUE;

    machinevoffset = (SWORD) 255 - memory[A_VSCROLL];
    if (vlock)
    {   uservoffset    = (SWORD) VLOCK_OFFSET;
        soffset[flip]  = 255 - VLOCK_OFFSET - memory[A_VSCROLL]; // can be negative
    } else
    {   uservoffset    = (SWORD) 255 - memory[A_VSCROLL];
        soffset[flip]  = 0;
    }

    if (memory[A_RESOLUTION] & 0x40) // lower screen on/off flag
    {   rowlimit = 26;
    } else
    {   rowlimit = 13;
    }
    hires = (memory[A_BGCOLOUR] & 0x80) >> 7; // %10000000 -> %00000001 (doublescan flag)

    psu &= ~(PSU_S);
    uviwrite(A_CHARLINE, 0xFF);
    breakdma();

    // top margin---------------------------------------------------------

    if (uservoffset)
    {   if (uservoffset < BOXHEIGHT)
        {   stopline = uservoffset;
        } else
        {   stopline = BOXHEIGHT;
        }
        for (y = 0; y < stopline; y++)
        {   if (flagline && (psu & PSU_F))
            {   theflag[y] = PSU_F;
                bgcolour[y] = bgc = 7 - (memory[A_BGCOLOUR] & 0x07);
            } else
            {   theflag[y] = 0;
                bgcolour[y] = bgc = memory[A_BGCOLOUR] & 0x07;
            }

            if (!vlock)
            {   slice = CPL - overcalc;
                cpu();
                CHECKINPUT;
            }

            // draw entire (blank) line
            for (x = 0; x < BOXWIDTH; x++)
            {   if (screen[x][y] != bgc)
                {   drawpixel(x, y, bgc);
    }   }   }   }
    if (vlock)
    {   slice = (CPL * machinevoffset) - overcalc;
        cpu();
        CHECKINPUT;
    }
    uviwrite(A_P1PADDLE,      0xFF); // "The potentiometer is set to $FF
    uviwrite(A_P2PADDLE,      0xFF); // on the trailing edge of VRST."
    // These have to be commented out for DORAEMON collision detection to work reliably
    // uviwrite(A_SPRITECOLLIDE, 0xFF); // "Each [bit] is reset to 1...at the end of the vertical blanking period."
    // uviwrite(A_BGCOLLIDE,     0xFF); // "Each [bit] is reset [to 1]...at the end of the vertical blanking period."

    // main (character) area----------------------------------------------

    rowc     =
    row      = 0;
    newband  = TRUE;
    y = uservoffset;

    for (rast = 0; rast < 262; rast++) // NTSC has 262 rastlines, value for true PAL would be 312?
    {   if (newband)
        {   // start of a new "band" (row) of characters (128*8 pixels in size)

            // BOWLING relies on boardmode being refreshed for each new "band"
            boardmode = memory[A_PITCH] & 0x80;
            if (boardmode)
            {   fgc1 = (memory[A_RESOLUTION] & 0x38) >> 3; // %00111000 -> %00000111
                fgc2 = (memory[A_BGCOLOUR]   & 0x38) >> 3; // %00111000 -> %00000111 colours of tile set 0
                bgc1 =  memory[A_RESOLUTION] & 0x07;       // %00000111
                bgc2 =  memory[A_BGCOLOUR]   & 0x07;       // %00000111              background colour
            }

            // REDCLASH relies on the row of characters being cached
            if (row <= 12)
            {   for (x = 0; x < 16; x++)
                {   rowbuf[x] = memory[A_OFFSET_UPPERSCREEN + (row << 4) + x];
            }   }
            else
            {   // row can be 26 or more, but rowbuf[] is not used in that case
                for (x = 0; x < 16; x++)
                {   rowbuf[x] = memory[0x1930               + (row << 4) + x];
            }   }

            hoffset = A_XMARGIN + ((memory[A_VOLUME] & 0xE0) >> 5); // 3 bits wide
            row++;

            newband = FALSE;
            if (memory[A_CHARLINE] != dma[(memory[A_BGCOLOUR] & 0x80) >> 7][rast >> 3])
            {   uviwrite(A_CHARLINE, dma[(memory[A_BGCOLOUR] & 0x80) >> 7][rast >> 3]);
                breakdma();
        }   }
        breakrastline();

        if (y < BOXHEIGHT)
        {   if (flagline && (psu & PSU_F))
            {   theflag[y] = PSU_F;
                bgc = bgcolour[y] = 7 - (memory[A_BGCOLOUR] & 0x07);
            } else
            {   theflag[y] = 0;
                bgc = bgcolour[y] =      memory[A_BGCOLOUR] & 0x07 ;
            }

            slice = CPL - overcalc;
            cpu();
            CHECKINPUT;

            // ** draw one Y-line of the display
            if (row <= rowlimit)
            {   // left margin
                for (x = 0; x < hoffset; x++)
                {   if (screen[x][y] != bgc)
                    {   drawpixel(x, y, bgc);
                }   }

                // centre area
                // assert(x == hoffset);
                // x = hoffset;
                blockmode = (memory[A_RESOLUTION] & 0x80) >> 7;
                for (column = 0; column < 16; column++)
                {   // ** draw one character (or block graphic) **

                    thechar = rowbuf[column];
                    if (thechar == 0xC0)
                    {   blockmode = 1;
                    } elif (thechar == 0x40)
                    {   blockmode = 0;
                    }

                    // get colours for this tile
                    if (boardmode)
                    {   if (theflag[y]) // or: if (flagline && (psu & PSU_F))
                        {   if (thechar & 0x40)
                            {   fgc = (UBYTE) (7 - fgc2);
                            } else
                            {   fgc = (UBYTE) (7 - fgc1);
                            }
                            if (thechar & 0x80)
                            {   bgc = (UBYTE) (7 - bgc2);
                            } else
                            {   bgc = (UBYTE) (7 - bgc1);
                        }   }
                        else
                        {   if (thechar & 0x40)
                            {   fgc =     fgc2;
                            } else
                            {   fgc =     fgc1;
                            }
                            if (thechar & 0x80)
                            {   bgc =     bgc2;
                            } else
                            {   bgc =     bgc1;
                    }   }   }
                    else
                    {   if (theflag[y]) // or: if (flagline && (psu & PSU_F))
                        {   fgc = 7
                                - ((thechar              &  0xC0) >> 5)  // %11000000 -> %00000110 (0, 2, 4 or 6)
                                - ((memory[A_BGCOLOUR] >>    3) &  1); // 0 or 1
                            bgc = 7 - (memory[A_BGCOLOUR] & 0x07);
                        } else
                        {   fgc = ((thechar              &  0xC0) >> 5)  // %11000000 -> %00000110 (0, 2, 4 or 6)
                                + ((memory[A_BGCOLOUR] >>    3) &  1); // 0 or 1
                            bgc =      memory[A_BGCOLOUR] & 0x07 ;
                    }   }

                    // get image data for this tile
                    thechar &= 0x3F;
                    if (!blockmode && thechar >= 56) // this is a UDC
                    {   imagedata = memory[A_OFFSET_SPRITES + (8 * (thechar - 56)) + rowc];
                    } else // built-in graphics
                    {   imagedata = pix[blockmode][thechar][rowc];
                    }

                    // draw one row of this tile (ie. 8*1)
                    for (xx = 0; xx < 8; xx++)
                    {   if (imagedata & (0x80 >> xx))
                        {   if (screen[x][y] != fgc)
                            {   drawpixel(x, y, fgc);
                        }   }
                        else
                        {   if (screen[x][y] != bgc)
                            {   drawpixel(x, y, bgc);
                        }   }
                        x++;
                }   }

                // right margin
                // in case we are in board mode
                // assert(x == hoffset + 128);
                do
                {   if (screen[x][y] != bgcolour[y])
                    {   drawpixel(x, y, bgcolour[y]);
                    }
                    x++;
                } while (x < BOXWIDTH);
            } else
            {   // bottom margin
                // draw entire (blank) line

                for (x = 0; x < BOXWIDTH; x++)
                {   if (screen[x][y] != bgc)
                    {   drawpixel(x, y, bgc);
        }   }   }   }
        else // this raster is not shown
        {   slice = CPL - overcalc;
            cpu();
            CHECKINPUT;
        }

        specialdma();
        if (hires || (rast & 1))
        {   if (rowc == 7)
            {   rowc = 0;
                newband = TRUE;
            } else rowc++;
        }

        y++;
    }

    // rastline is 0 (262), Sense bit is set, CHARLINE is $FD
    drawallsprites(); // must not be done at SENSELINE, eg. REDCLASH
    collidesprites(); // must not be done at SENSELINE, eg. ROBOTKIL, ESCAPE

    psu &= ~(PSU_S); // clear Sense bit. Necessary, eg. CIRCUS
    uviwrite(A_CHARLINE, 0xFF); // possibly unnecessary or wrong?
    breakdma();

    // actual diagonal (bottom-right to top-left) beam retrace is here
    slice = (CPL * 3) - overcalc;
    cpu();
    CHECKINPUT;

    // Currently, frame skipping only affects warp mode when "limit
    // refresh rate" is off.

    if
    (   (!warp || !limitrefresh)
     && (frames % frameskip == 0)
    )
    {   updatescreen();
    }

    flip ^= 1; // means: if (flip) flip = 0; else flip = 1;
    frames++;

    inframe = FALSE;
    if (crippled)
    {   uncripple();
    }

    if (recmode == RECMODE_PLAY && offset >= recsize)
    {   if (loop)
        {   macro_restartplayback();
        } else
        {   macro_stop();
    }   }

    if (runtoframe)
    {   runtoframe = FALSE;
        OPENCONSOLE;
        printf("Reached next frame (%ld).\n\n", frames);
        REACTIVATE;
        pause();
}   }

MODULE void drawallsprites(void)
{   // cache sprite variables
     tall[flip][0] = 2 - ((memory[A_SPRITES01CTRL] & 0x80) >> 7); // %10000000 -> %00000001
     tall[flip][1] = 2 - ((memory[A_SPRITES01CTRL] & 0x40) >> 6); // %01000000 -> %00000001
     tall[flip][2] = 2 - ((memory[A_SPRITES23CTRL] & 0x80) >> 7); // %10000000 -> %00000001
     tall[flip][3] = 2 - ((memory[A_SPRITES23CTRL] & 0x40) >> 6); // %01000000 -> %00000001
    spritex[flip][0] =               memory[A_SPRITE0X] - 43 + A_XMARGIN;
    spritey[flip][0] = (SWORD) 256 - memory[A_SPRITE0Y] - soffset[flip];
    spritex[flip][1] =               memory[A_SPRITE1X] - 43 + A_XMARGIN;
    spritey[flip][1] = (SWORD) 256 - memory[A_SPRITE1Y] - soffset[flip];
    spritex[flip][2] =               memory[A_SPRITE2X] - 43 + A_XMARGIN;
    spritey[flip][2] = (SWORD) 256 - memory[A_SPRITE2Y] - soffset[flip];
    spritex[flip][3] =               memory[A_SPRITE3X] - 43 + A_XMARGIN;
    spritey[flip][3] = (SWORD) 256 - memory[A_SPRITE3Y] - soffset[flip];

    t = 0xFF;
    for (whichsprite = 0; whichsprite < 4; whichsprite++)
    {   // cache sprite background flag pin state
        for (y = 0; y < tall[flip][whichsprite] * 8; y++)
        {   yy = spritey[flip][whichsprite] + y;
            if (yy >= BOXHEIGHT)
            {   break;
            }
            switch(whichsprite)
            {
            case 0:
                if (theflag[yy])
                {   spritefgc[flip][whichsprite][y] = (UBYTE) 7 - ((memory[A_SPRITES01CTRL] & 0x38) >> 3); // %00111000 -> %00000111
                } else
                {   spritefgc[flip][whichsprite][y] = (UBYTE)     ((memory[A_SPRITES01CTRL] & 0x38) >> 3); // %00111000 -> %00000111
                }
            acase 1:
                if (theflag[yy])
                {   spritefgc[flip][whichsprite][y] = (UBYTE) 7 - ( memory[A_SPRITES01CTRL] & 0x07      ); // %00000111 -> %00000111
                } else
                {   spritefgc[flip][whichsprite][y] = (UBYTE)     ( memory[A_SPRITES01CTRL] & 0x07      ); // %00000111 -> %00000111
                }
            acase 2:
                if (theflag[yy])
                {   spritefgc[flip][whichsprite][y] = (UBYTE) 7 - ((memory[A_SPRITES23CTRL] & 0x38) >> 3); // %00111000 -> %00000111
                } else
                {   spritefgc[flip][whichsprite][y] = (UBYTE)     ((memory[A_SPRITES23CTRL] & 0x38) >> 3); // %00111000 -> %00000111
                }
            acase 3:
                if (theflag[yy])
                {   spritefgc[flip][whichsprite][y] = (UBYTE) 7 - ( memory[A_SPRITES23CTRL] & 0x07      ); // %00000111 -> %00000111
                } else
                {   spritefgc[flip][whichsprite][y] = (UBYTE)     ( memory[A_SPRITES23CTRL] & 0x07      ); // %00000111 -> %00000111
                }
            adefault:
                // assert(0);
            break;
        }   }

        // cache sprite imagery
        for (y = 0; y < 8; y++)
        {   image[flip][whichsprite][y] = memory[A_OFFSET_SPRITES + (whichsprite * 8) + y];
        }

/* bgcolour[y] is sampled (written) at the start of the right margin...
   but really we could have several bgcolours for the one rastline, so the below reliance on
   bgcolour[y] is somewhat dodgy... */

        // check sprite position
        if (memory[A_SPRITE0X + (2 * whichsprite)] > 227)
        {   // "If the HC is set to >227, the object is effectively
            // removed from the video field."
            skip[flip][whichsprite] = TRUE;
        } else
        {   skip[flip][whichsprite] = FALSE;

            // detect sprite-background collisions
            // it is important that this is done BEFORE drawing any sprites
            if (collisions)
            {   x = spritex[flip][whichsprite];
                y = spritey[flip][whichsprite];
                if (tall[flip][whichsprite] == 1)
                {   for (xx = 0; xx < 8; xx++)
                    {   if (x >= 0 && x < BOXWIDTH)
                        {   for (yy = 0; yy < 8; yy++)
                            {   if (y >= 0 && y < BOXHEIGHT)
                                {   imagedata = image[flip][whichsprite][yy];

                                    if
                                    (   (imagedata & (0x80 >> xx)) // if this sprite-pixel is set
                                     && screen[x][y] != bgcolour[y]
                                    )
                                    {   t &= ~(1 << whichsprite);
                                        break;
                                    }
                                }
                                y++;
                            }
                            y -= 8;
                        }
                        x++;
                }   }
                else
                {   // assert(tall[flip][whichsprite] == 2);
                    for (xx = 0; xx < 8; xx++)
                    {   if (x >= 0 && x < BOXWIDTH)
                        {   for (yy = 0; yy < 8; yy++)
                            {   if (y >= 0 && y < BOXHEIGHT)
                                {   imagedata = image[flip][whichsprite][yy];

                                    if (imagedata & (0x80 >> xx)) // if this sprite-pixel is set
                                    {   if (screen[x][y] != bgcolour[y])
                                        {   t &= ~(1 << whichsprite);
                                            break;
                                        }
                                        if (y < BOXHEIGHT - 1)
                                        {   if (screen[x][y + 1] != bgcolour[y + 1])
                                            {   t &= ~(1 << whichsprite);
                                                break;
                                }   }   }   }
                                y += 2;
                            }
                            y -= 16;
                        }
                        x++;
    }   }   }   }   }
    uviwrite(A_BGCOLLIDE, t);

    if (frames && demultiplex) // if first frame, we won't have valid data! :-(
    {   if (flip)
        {   drawsprites(0);
        } else
        {   drawsprites(1);
    }   }
    drawsprites(flip);
}

MODULE __inline void collidesprites(void)
{   /* This doesn't use screen[], thus it is safe at any point in the frame. */

    t = 0xFF;
    if (collisions)
    {   if (sprite_collision(0, 1)) t &=  ~1; // AND with %11111110 (clear bit 0)
        if (sprite_collision(0, 2)) t &=  ~2; // AND with %11111101 (clear bit 1)
        if (sprite_collision(0, 3)) t &=  ~4; // AND with %11111011 (clear bit 2)
        if (sprite_collision(1, 2)) t &=  ~8; // AND with %11110111 (clear bit 3)
        if (sprite_collision(1, 3)) t &= ~16; // AND with %11101111 (clear bit 4)
        if (sprite_collision(2, 3)) t &= ~32; // AND with %11011111 (clear bit 5)
    }
    uviwrite(A_SPRITECOLLIDE, t);
}

MODULE void drawsprites(SLONG flipper)
{   for (whichsprite = 0; whichsprite < 4; whichsprite++)
    {   if (!skip[flipper][whichsprite])
        {   x = spritex[flipper][whichsprite];
            y = spritey[flipper][whichsprite];

            if (tall[flipper][whichsprite] == 1)
            {   for (yy = 0; yy < 8; yy++)
                {   if (y >= 0 && y < BOXHEIGHT)
                    {   imagedata =     image[flipper][whichsprite][yy];
                        sprcolour = spritefgc[flipper][whichsprite][yy];

                        for (xx = 0; xx < 8; xx++)
                        {   if (imagedata & (0x80 >> xx)) // if this x-bit is set
                            {   if (x >= 0 && x < BOXWIDTH)
                                {   if (screen[x][y] != sprcolour)
                                    {   drawpixel(x, y, sprcolour);
                            }   }   }
                            x++;
                        }
                        x -= 8;
                    }
                    y++;
            }   }
            else
            {   for (yy = 0; yy < 8; yy++) // for each y-line of the sprite
                {   if (y >= 0 && y < BOXHEIGHT)
                    {   imagedata =     image[flipper][whichsprite][yy];
                        sprcolour = spritefgc[flipper][whichsprite][yy * 2];

                        for (xx = 0; xx < 8; xx++)
                        {   if (imagedata & (0x80 >> xx)) // if this x-bit is set
                            {   if (x >= 0 && x < BOXWIDTH)
                                {   if (screen[x][y] != sprcolour)
                                    {   drawpixel(x, y, sprcolour);
                                    }

                                    if (y < BOXHEIGHT - 1)
                                    {   if (screen[x][y + 1] != sprcolour)
                                        {   drawpixel(x, y + 1, sprcolour);
                            }   }   }   }
                            x++;
                        } // rof (end for)
                        x -= 8;
                    }
                    y += 2;
}   }   }   }   }

/* Each raster is about  12x 16  =  192 clock periods.
   Each frame  is about 192x262/2       clock periods.
*/

MODULE void a_playerinput(UBYTE source, UBYTE dest)
{   // dest is which side you want to set the registers of.
    // source is which side you want to use to do it.

    if (recmode == RECMODE_PLAY)
    {   if (dest == 0)
        {
#ifdef AMIGA
            uviwrite(A_P1PADDLE    , IOBuffer[offset++]);
#endif
#ifdef WIN32
            uviwrite(A_P1PADDLE    , IOBuffer[offset]);
            if (psu & PSU_F) // paddle interpolation bit
            {   sy[dest] = IOBuffer[offset++];
            } else
            {   sx[dest] = IOBuffer[offset++];
            }
#endif
            uviwrite(A_P1LEFTKEYS  , IOBuffer[offset++]);
            uviwrite(A_P1MIDDLEKEYS, IOBuffer[offset++]);
            uviwrite(A_P1RIGHTKEYS , IOBuffer[offset++]);
            uviwrite(A_P1PALLADIUM , IOBuffer[offset++]);
        } else
        {   // assert(dest == 1);

#ifdef AMIGA
            uviwrite(A_P2PADDLE    , IOBuffer[offset++]);
#endif
#ifdef WIN32
            uviwrite(A_P2PADDLE    , IOBuffer[offset]);
            if (psu & PSU_F) // paddle interpolation bit
            {   sy[dest] = IOBuffer[offset++];
            } else
            {   sx[dest] = IOBuffer[offset++];
            }
#endif
            uviwrite(A_P2LEFTKEYS  , IOBuffer[offset++]);
            uviwrite(A_P2MIDDLEKEYS, IOBuffer[offset++]);
            uviwrite(A_P2RIGHTKEYS , IOBuffer[offset++]);
            uviwrite(A_P2PALLADIUM , IOBuffer[offset++]);
    }   }
    else
    {   // assert(recmode == RECMODE_NORMAL || recmode == RECMODE_RECORD);
        xx   =
        yy   = 0;

        result = ReadJoystick((UWORD) (1 - source));
        if   (result & JOYUP)                 yy = -1;
        elif (result & JOYDOWN)               yy =  1;
        if   (result & JOYLEFT)               xx = -1;
        elif (result & JOYRIGHT)              xx =  1;

        if   (KeyDown(a_keypads[source][17])) yy = -1;
        elif (KeyDown(a_keypads[source][18])) yy =  1;
        if   (KeyDown(a_keypads[source][19])) xx = -1;
        elif (KeyDown(a_keypads[source][20])) xx =  1;

        if (analog)
        {   if   (xx  == -1) { ax[dest] -= 4; if (ax[dest] <   0) ax[dest] =   0; }
            elif (xx  ==  1) { ax[dest] += 4; if (ax[dest] > 255) ax[dest] = 255; }
            if   (yy  == -1) { ay[dest] -= 4; if (ay[dest] <   0) ay[dest] =   0; }
            elif (yy  ==  1) { ay[dest] += 4; if (ay[dest] > 255) ay[dest] = 255; }
            bx = (UBYTE) ax[dest];
            by = (UBYTE) ay[dest];
        } else
        {   if (xx == -1) bx = 0; elif (xx == 1) bx = 254; else bx = 112;
            if (yy == -1) by = 0; elif (yy == 1) by = 254; else by = 112;
        }

        if (memory[A_BGCOLOUR] & 0x40) // paddle interpolation bit
        {   uviwrite((int) (A_P1PADDLE - dest), bx);
#ifdef WIN32
            sx[dest] = bx;
#endif
        } else
        {   uviwrite((int) (A_P1PADDLE - dest), by);
#ifdef WIN32
            sy[dest] = by;
#endif
        }

        // "primary" firebutton is always 2 for Arcadia

        t  = (UBYTE) (8 * KeyDown(a_keypads[source][ 1]));
#ifdef WIN32
        t += (4 * KeyDown(a_keypads[source][ 4]));
#endif
#ifdef AMIGA
        if
        (   KeyDown(a_keypads[source][ 4])
         || (source == 1 && KeyDown(SCAN_ND))
         || (result & JOYFIRE2)
        )
        {   t += 4;
        }
#endif
        t += (2 * KeyDown(a_keypads[source][ 7]));
        t +=      KeyDown(a_keypads[source][10]) ;
        uviwrite(A_P1LEFTKEYS + (dest * 4), t);

        if
        (
            (autofire && ((frames % totalframes) < downframes)) // ie. it is held down for 4 frames, then up for 12 frames
         || KeyDown(a_keypads[source][0])
         || KeyDown(a_keypads[source][2])
         || (result & JOYFIRE1)
        )
        {   t = 8;
        } else
        {   t = 0;
        }
        t += (4 * KeyDown(a_keypads[source][ 5]));
        t += (2 * KeyDown(a_keypads[source][ 8]));
        if
        (   KeyDown(a_keypads[source][11])
         || (result & JOYFIRE4)
        )
        {   t++;
        }
        uviwrite(A_P1MIDDLEKEYS + (dest * 4), t);

        if
        (   KeyDown(a_keypads[source][ 3])
         || (result & JOYFIRE3)
        )
        {   t = 8;
        } else
        {   t = 0;
        }
        t += (4 * KeyDown(a_keypads[source][ 6]));
        t += (2 * KeyDown(a_keypads[source][ 9]));
        t +=      KeyDown(a_keypads[source][12]) ;
        uviwrite(A_P1RIGHTKEYS + (dest * 4), t);

        t  = (UBYTE) (8 * KeyDown(a_keypads[source][13]));
        t += (4 * KeyDown(a_keypads[source][14]));
        t += (2 * KeyDown(a_keypads[source][15]));
        t +=      KeyDown(a_keypads[source][16]) ;
        uviwrite(A_P1PALLADIUM + (dest * 4), t);
}   }

MODULE __inline void a_emuinput(void)
{   readkybd();

    // must always do a_playerinput(foo, 0) then a_playerinput(foo, 1).

#ifdef NIGGER
if (recmode == RECMODE_PLAY) {
OPENCONSOLE;
printf("Playback %ld: P1: %ld, %ld, %ld, %ld, %ld! P2: %ld %ld, %ld, %ld, %ld!\n",
frames,
IOBuffer[offset],
IOBuffer[offset + 1],
IOBuffer[offset + 2],
IOBuffer[offset + 3],
IOBuffer[offset + 4],
IOBuffer[offset + 5],
IOBuffer[offset + 6],
IOBuffer[offset + 7],
IOBuffer[offset + 8],
IOBuffer[offset + 9],
IOBuffer[offset + 10]
);
}
#endif

    if (swapped)
    {   a_playerinput(1, 0);
        a_playerinput(0, 1);
    } else
    {   a_playerinput(0, 0);
        a_playerinput(1, 1);
    }

    if (recmode == RECMODE_RECORD)
    {   // assert(MacroHandle);

        OutputBuffer[0]  = memory[A_P1PADDLE    ];
        OutputBuffer[1]  = memory[A_P1LEFTKEYS  ];
        OutputBuffer[2]  = memory[A_P1MIDDLEKEYS];
        OutputBuffer[3]  = memory[A_P1RIGHTKEYS ];
        OutputBuffer[4]  = memory[A_P1PALLADIUM ];
        OutputBuffer[5]  = memory[A_P2PADDLE    ];
        OutputBuffer[6]  = memory[A_P2LEFTKEYS  ];
        OutputBuffer[7]  = memory[A_P2MIDDLEKEYS];
        OutputBuffer[8]  = memory[A_P2RIGHTKEYS ];
        OutputBuffer[9]  = memory[A_P2PALLADIUM ];
        OutputBuffer[10] = memory[A_CONSOLE     ];

        DISCARD fwrite(OutputBuffer, 11, 1, MacroHandle); // should really check return code

        // 11 bytes per frame * 60 frames per second = 660 bytes/sec.
}   }

MODULE int sprite_collision(int first, int second)
{   int        b1,
               b2,
               y;
    signed int x;

    // check whether they are close enough
    if
    (   spritex[flip][first] <= spritex[flip][second] - 8
     || spritex[flip][first] >= spritex[flip][second] + 8
    )
    {   return FALSE;
    }

    // Undrawn sprites are not collision-checked! This behaviour
    // is needed for eg. DORAEMON.
    if (skip[flip][first] || skip[flip][second])
    {   return FALSE;
    }

    // calculate how much further to the right the first is
    // than the second, eg.
    // x <  0: first  is left, second is right
    // x == 0: both are at same X-coordinate
    // x >  0: second is left, first  is right
    x = (signed int) (spritex[flip][first] - spritex[flip][second]);

    for (y = 0; y < 8; y++)
    {   // check whether we are above the top of the second sprite
        if
        (   spritey[flip][first] + (y * tall[flip][first])
         <  spritey[flip][second]
        )
        {   continue;
        }
        // check whether we are below the bottom of the second sprite
        if
        (   spritey[flip][first] + (y * tall[flip][first])
         >= spritey[flip][second] + (8 * tall[flip][second])
        )
        {   return FALSE;
        }

/*      Examples:

        195 00000000
        196 11111111
        197 22222222
        198 33333333
        199 44444444
        200 55555555    00000000
        201 66666666    00000000
        202 77777777    11111111
        203             11111111
        204             22222222
        205             22222222 etc.

        yy would be -5.

        195 00000000
        196 00000000
        197 11111111
        198 11111111
        199 22222222
        200 22222222    00000000
        201 33333333    11111111
        202 33333333    22222222
        203 44444444    33333333
        204 44444444    44444444
        205 55555555    55555555 etc.

        yy would be -5.

        195             00000000
        196             11111111
        197             22222222
        198             33333333
        199             44444444
        200 00000000    55555555
        201 00000000    66666666
        202 11111111    77777777
        203 11111111
        204 22222222
        205 22222222 etc.

        yy would be 5.

        195             00000000
        196             00000000
        197             11111111
        198             11111111
        199             22222222
        200 00000000    22222222
        201 00000000    33333333
        202 11111111    33333333
        203 11111111    44444444
        204 22222222    44444444
        205 22222222    55555555 etc.

        yy would be 5. */

        yy = spritey[flip][first] - spritey[flip][second];
        b1 = image[flip][first][y];

        if (tall[flip][first] == tall[flip][second]) // same vs. same
        {   b2 = image[flip][second][yy + y];
        } elif (tall[flip][first] == 1 && tall[flip][second] == 2) // short vs. tall
        {   b2 = image[flip][second][(yy + y) / 2];
        } else
        {   // assert(tall[flip][first] == 2 && tall[flip][second] == 1); // tall vs. short
            b2 = image[flip][second][yy + (y / 2)];
        }

        if (x < 0) b2 >>= -x;
        if (x > 0) b1 >>= x;
        if (b1 & b2)
        {   return TRUE;
    }   }

    return FALSE;
}

/* Note that ALIENINV uses sprite-background collision detection,
not sprite-sprite collision detection, for the UFO (aka mothership).
The only sprite-sprite collisions are for players colliding with the
enemy bombs.

CIRCUS waits until CHARLINE == $FD, then immediately checks
SPRITECOLLIDE.

PLEIADES doesn't seem to use the SPRITECOLLIDE or BGCOLLIDE registers
at all. Thus, turning off collision detection will have no effect on
this game.

Games with twinkling starfields:
    SATTAC?
    SVULTURE
    THE-END
and probably others.

Games making use of multiplexing for sprites and/or characters:
    DR-SLUMP         2 on, 2 off
    FUNKFISH         1 on, 1 off. Works.
    MACROSS
     outside ship:   2 on, 2 off
     inside ship:    about 4 on, about 4 off
    MISSILEW         2 on, 2 off. Not sprites.
    REDCLASH         1 on, 1 off. Works.
    R2DTANK          1 on, 2 off
    SPACEAT?         1 on, 1 off. Works.
    SMISSION
     uppermost:      infinite on, 0 off.
     lowermost:      1 on, 1 off. Not sprites?
    SRAIDERS         1 on, 1 off. Not sprites.
    SSQUADRO         1 on, 1 off. Not sprites?
    SPIDERS          1 on, 3 off
    SVULTURE
     player bullets: 1 on, 7 off
     alien bullets:  1 on, 4 off
    TENNIS           1 on, 1 off. Works.
    TURTLES          1 on, 1 off. Works.
*/

MODULE __inline void specialdma(void)
{   switch(rast)
    {
    case INPUTLINE: // 207
        a_emuinput();
    acase LATCHLINE: // 225
        if (recmode == RECMODE_PLAY)
        {   uviwrite(A_CONSOLE, IOBuffer[offset++]);
        } else
        {   if (KeyDown(SCAN_F1) || latch_start > 0)
            {   if (latch_start > 0)
                {   latch_start--;
                }
                t = 1;
            } else
            {   t = 0;
            }
            if (KeyDown(SCAN_F2) || latch_a     > 0)
            {   latch_a = 0;
                t += 2;
            }
            if (KeyDown(SCAN_F3) || latch_b     > 0)
            {   latch_b = 0;
                t += 4;
            }
            uviwrite(A_CONSOLE, t);
        }
    acase SENSELINE: // 240
        psu |= PSU_S;
    adefault:
    break;
}   }

MODULE __inline void breakdma(void)
{   if (runtodma)
    {   runtodma = FALSE;
        OPENCONSOLE;
        printf("Reached next DMA ($%X).\n\n", memory[A_CHARLINE]);
        REACTIVATE;
        runto();
}   }

MODULE __inline void breakrastline(void)
{   if (runtorastline)
    {   runtorastline = FALSE;
        OPENCONSOLE;
        printf("Reached next raster line (%ld).\n\n", rast);
        REACTIVATE;
        runto();
    }
    if (rast == rastn)
    {   OPENCONSOLE;
        printf("Reached raster line %ld.\n\n", rast);
        REACTIVATE;
        runto();
}   }

MODULE void uviwrite(int address, UBYTE data)
{   // assert(address < 32768);

    /* Writes to memory from the UVI always go through this
    routine (so that watchpoints work properly). */

    if (address == wp && data != memory[wp])
    {   getfriendly(wp);
        OPENCONSOLE;
        printf("UVI is writing $%X to %s at raster %ld!\n\n", data, (char *) friendly, rast); // just cast for lint
        REACTIVATE;
        runto();
    }

    memory[address] = data;
}

#ifdef WIN32

MODULE __inline void drawpixel(SLONG x, SLONG y, UBYTE colour)
{   // assert(x >= 0);
    // assert(x < BOXWIDTH);
    // assert(y >= 0);
    // assert(y < BOXHEIGHT);
    // assert(colour >= 0 && colour <= 7);

    screen[x][y] = colour;

#ifdef MODA_SHALLOW
    display[x + (y * EVENBOXWIDTH)] = colour;
#else
    display[x + (y * BOXWIDTH)] = pens[colour];
#endif

}

#endif

#ifdef AMIGA
#ifndef USEFUNCPTRS

MODULE __inline void drawpixel(SLONG x, SLONG y, UBYTE colour)
{   // assert(x >= 0);
    // assert(x < BOXWIDTH);
    // assert(y >= 0);
    // assert(y < BOXHEIGHT);
    // assert(colour >= 0 && colour <= 7);

    screen[x][y] = colour;

    if (fullscreen) // 2*1
    {   // could use pens[colour] but that would be slower

        px = x << 1; // px = x * 2;

        *(lineptr[y] + px++) = colour; // 1,1
        *(lineptr[y] + px)   = colour; // 2,1
    } else
    {   switch(size)
        {
        case 1:
            if (wide == 1) // 1*1
            {   *(lineptr[y     ] + x)    = pens[colour]; // 1,1
            } else
            {   // assert(wide == 2); // 2*1

                px = x << 1; // px = x * 2;

                *(lineptr[y     ] + px++) = pens[colour]; // 1,1
                *(lineptr[y     ] + px)   = pens[colour]; // 2,1
            }
        acase 2:
            py = y << 1; // py = y * 2;

            if (wide == 1) // 2*2
            {   px = x << 1; // px = x * 2;

                *(lineptr[py    ] + px)   = pens[colour]; // 1,1
                *(lineptr[py + 1] + px++) = pens[colour]; // 1,2
                *(lineptr[py    ] + px)   = pens[colour]; // 2,1
                *(lineptr[py + 1] + px)   = pens[colour]; // 2,2
            } else
            {   // assert(wide == 2); // 4*2

                px = x << 2; // px = x * 4;

                *(lineptr[py    ] + px)   = pens[colour]; // 1,1
                *(lineptr[py + 1] + px++) = pens[colour]; // 1,2
                *(lineptr[py    ] + px)   = pens[colour]; // 2,1
                *(lineptr[py + 1] + px++) = pens[colour]; // 2,2
                *(lineptr[py    ] + px)   = pens[colour]; // 3,1
                *(lineptr[py + 1] + px++) = pens[colour]; // 3,2
                *(lineptr[py    ] + px)   = pens[colour]; // 4,1
                *(lineptr[py + 1] + px)   = pens[colour]; // 4,2
            }
        adefault:
            // assert(size == 3);

            px = x * 3 * wide;
            py = y * 3;

                *(lineptr[py    ] + px)   = pens[colour]; // 1,1
                *(lineptr[py + 1] + px)   = pens[colour]; // 1,2
                *(lineptr[py + 2] + px++) = pens[colour]; // 1,3
                *(lineptr[py    ] + px)   = pens[colour]; // 2,1
                *(lineptr[py + 1] + px)   = pens[colour]; // 2,2
                *(lineptr[py + 2] + px++) = pens[colour]; // 2,3
                *(lineptr[py    ] + px)   = pens[colour]; // 3,1
                *(lineptr[py + 1] + px)   = pens[colour]; // 3,2
                *(lineptr[py + 2] + px++) = pens[colour]; // 3,3

            if (wide == 1) return;

                *(lineptr[py    ] + px)   = pens[colour]; // 4,1
                *(lineptr[py + 1] + px)   = pens[colour]; // 4,2
                *(lineptr[py + 2] + px++) = pens[colour]; // 4,3
                *(lineptr[py    ] + px)   = pens[colour]; // 5,1
                *(lineptr[py + 1] + px)   = pens[colour]; // 5,2
                *(lineptr[py + 2] + px++) = pens[colour]; // 5,3
                *(lineptr[py    ] + px)   = pens[colour]; // 6,1
                *(lineptr[py + 1] + px)   = pens[colour]; // 6,2
                *(lineptr[py + 2] + px)   = pens[colour]; // 6,3

        break;
}   }   }

#endif
#endif

